<?php
declare(strict_types=1);

// progress.php
// خروجی زنده پیشرفت دانلود برای نمایش اسپیدومترها و درصد دقیق

// هدرها برای JSON و جلوگیری از کش
header('Content-Type: application/json; charset=utf-8');
header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
header('Pragma: no-cache');

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['error' => 'Method not allowed']);
    exit;
}

// دریافت و ایمن‌سازی شناسه
$progressId = trim($_GET['id'] ?? '');
if ($progressId === '' || !preg_match('~^[A-Za-z0-9._-]+$~', $progressId)) {
    http_response_code(400);
    echo json_encode(['error' => 'Missing or invalid id']);
    exit;
}

// مسیر فایل وضعیت دانلود
$progressDir  = __DIR__ . '/storage/progress';
$progressFile = $progressDir . '/' . basename($progressId) . '.json';

// کمکی: فرمت خوانا برای انسان
function humanSize(int $bytes): string {
    if ($bytes < 1024) return $bytes . ' B';
    $kb = $bytes / 1024;
    if ($kb < 1024) return number_format($kb, 2) . ' KB';
    $mb = $kb / 1024;
    if ($mb < 1024) return number_format($mb, 2) . ' MB';
    $gb = $mb / 1024;
    return number_format($gb, 2) . ' GB';
}

// خروجی حداقلی در صورت نبود فایل
if (!is_file($progressFile)) {
    echo json_encode([
        'downloaded'      => 0,
        'total'           => 0,
        'percent'         => 0,
        'speed'           => 0,           // KB/s
        'eta_sec'         => null,        // زمان باقی‌مانده (ثانیه) اگر قابل محاسبه باشد
        'completed'       => false,
        'downloaded_str'  => '0 B',
        'total_str'       => '0 B',
        'updated_at'      => microtime(true),
    ]);
    exit;
}

// خواندن فایل با رتری کوتاه برای جلوگیری از نیمه‌نوشته بودن
$attempts = 0;
$dataRaw  = false;
while ($attempts < 3) {
    $dataRaw = @file_get_contents($progressFile);
    if ($dataRaw !== false && $dataRaw !== '') break;
    usleep(100 * 1000); // 100ms
    $attempts++;
}
if ($dataRaw === false || $dataRaw === '') {
    http_response_code(500);
    echo json_encode(['error' => 'Cannot read progress file']);
    exit;
}

// پارس JSON
$st = json_decode($dataRaw, true);
if (!is_array($st)) {
    http_response_code(500);
    echo json_encode(['error' => 'Invalid progress format']);
    exit;
}

// مقادیر پایه
$downloaded    = (int)($st['downloaded'] ?? 0);
$total         = (int)($st['total'] ?? 0);
$updatedAt     = isset($st['updated_at']) ? (float)$st['updated_at'] : microtime(true);
$lastTickTime  = isset($st['lastTickTime']) ? (float)$st['lastTickTime'] : 0.0;
$lastTickBytes = isset($st['lastTickBytes']) ? (int)$st['lastTickBytes'] : 0;

// محاسبه سرعت لحظه‌ای (KB/s)
$deltaT    = max(0.001, $updatedAt - $lastTickTime); // جلوگیری از تقسیم بر صفر
$deltaB    = max(0, $downloaded - $lastTickBytes);
$speedKBs  = ($deltaB / 1024) / $deltaT;             // KB/s

// درصد پیشرفت
$percent = 0;
if ($total > 0) {
    $percent = (int)round(($downloaded / $total) * 100);
    $percent = max(0, min(100, $percent));
}

// ETA (زمان باقی‌مانده به ثانیه) اگر قابل محاسبه باشد
$eta = null;
if ($total > 0 && $downloaded > 0 && $speedKBs > 0) {
    $remainingBytes = max(0, $total - $downloaded);
    // سرعت بر حسب B/s: speedKBs * 1024
    $eta = (int)round($remainingBytes / ($speedKBs * 1024));
}

// رشته‌های خوانا
$downloadedStr = humanSize($downloaded);
$totalStr      = humanSize($total);

// وضعیت تکمیل
$completed = ($total > 0 && $downloaded >= $total) || ($percent >= 100);

// خروجی JSON نهایی
echo json_encode([
    'downloaded'      => $downloaded,
    'total'           => $total,
    'percent'         => $percent,
    'speed'           => (float)number_format($speedKBs, 2, '.', ''), // KB/s
    'eta_sec'         => $eta,
    'completed'       => $completed,
    'downloaded_str'  => $downloadedStr,
    'total_str'       => $totalStr,
    'updated_at'      => $updatedAt,
], JSON_UNESCAPED_UNICODE);